//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2006 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using WeifenLuo.WinFormsUI;


namespace SID_monitor
{
    public partial class DateDockablePanel : DockContent
    {
        public DateDockablePanel()
        {
            InitializeComponent();

            // Graph Date Selection
            this.radioButtonGraphToday.Checked = true;
            this.radioButtonLast24Hours.Checked = false;
            this.radioButtonSelectDate.Checked = false;
            this.radioButtonSelectRange.Checked = false;
            this.monthCalendarGraphDate.SetDate(DateTime.UtcNow.Date);
            this.monthCalendarGraphDate.Enabled = false;

            this.dateTimePickerStartTime.Value = DateTime.UtcNow.Date;
            this.dateTimePickerEndTime.Value = DateTime.UtcNow.AddDays(1).Date;
            this.labelRefreshGraph.Visible = false;

        }

        #region Properties
        public bool GraphTodayChecked
        {
            get
            {
                return this.radioButtonGraphToday.Checked;
            }
            set
            {
                this.radioButtonGraphToday.Checked = value;
            }
        }

        public bool GraphLast24HoursChecked
        {
            get
            {
                return this.radioButtonLast24Hours.Checked;
            }
            set
            {
                this.radioButtonLast24Hours.Checked = value;
            }
        }

        public bool GraphSelectDateChecked
        {
            get
            {
                return this.radioButtonSelectDate.Checked;
            }
            set
            {
                this.radioButtonSelectDate.Checked = value;
            }
        }

        public bool GraphSelectRangeChecked
        {
            get
            {
                return this.radioButtonSelectRange.Checked;
            }
            set
            {
                this.radioButtonSelectRange.Checked = value;
            }
        }

        public DateTime MonthCalendarSelectionStart
        {
            get
            {
                return this.monthCalendarGraphDate.SelectionStart;
            }
            set
            {
                this.monthCalendarGraphDate.SelectionStart = value;
            }
        }

        public DateTime MonthCalendarSelectionEnd
        {
            get
            {
                return this.monthCalendarGraphDate.SelectionEnd;
            }
            set
            {
                this.monthCalendarGraphDate.SelectionEnd = value;
            }
        }

        public DateTime DateTimePickerStart
        {
            get
            {
                return this.dateTimePickerStartTime.Value;
            }
            set
            {
                this.dateTimePickerStartTime.Value = value;
            }
        }

        public DateTime DateTimePickerEnd
        {
            get
            {
                return this.dateTimePickerEndTime.Value;
            }
            set
            {
                this.dateTimePickerEndTime.Value = value;
            }
        }

        public string GraphLastUpdateText
        {
            get
            {
                return this.labelLastGraphUpdateValue.Text;
            }
            set
            {
                this.labelLastGraphUpdateValue.Text = value;
            }
        }

        public bool LabelRefreshGraphVisible
        {
            get
            {
                return this.labelRefreshGraph.Visible;
            }
            set
            {
                this.labelRefreshGraph.Visible = value;
            }
        }

#endregion

        #region Date Pane Methods
        /// <summary>
        /// Sets the today date of the month picker to the UTC date
        /// </summary>
        private void DateDockablePanel_Paint(object sender, PaintEventArgs e)
        {
            this.monthCalendarGraphDate.MaxDate = DateTime.UtcNow.Date;
            this.monthCalendarGraphDate.MinDate = this.monthCalendarGraphDate.MaxDate.AddMonths(-1);
            this.monthCalendarGraphDate.TodayDate = this.monthCalendarGraphDate.MaxDate;

            this.dateTimePickerStartTime.MaxDate = DateTime.UtcNow.Date.AddDays(1);
            this.dateTimePickerStartTime.MinDate = DateTime.UtcNow.Date.AddMonths(-1);

            this.dateTimePickerEndTime.MaxDate = this.dateTimePickerStartTime.MaxDate;
            this.dateTimePickerEndTime.MinDate = this.dateTimePickerStartTime.MinDate;

        }
        #endregion

        #region Date/Time Selection Methods

        private void buttonGraphUpdateNow_Click(object sender, EventArgs e)
        {
            OnGraphUpdateRequested(e);
        }

        private void radioButtonGraphToday_CheckedChanged(object sender, EventArgs e)
        {
            if (this.radioButtonGraphToday.Checked == true)
            {
                OnGraphUpdateRequested(e);
            }
        }

        private void radioButtonLast24Hours_CheckedChanged(object sender, EventArgs e)
        {
            if (this.radioButtonLast24Hours.Checked == true)
            {
                OnGraphUpdateRequested(e);
            }
        }

        private void radioButtonSelectRange_CheckedChanged(object sender, EventArgs e)
        {
            if (this.radioButtonSelectRange.Checked == true)
            {
                this.dateTimePickerStartTime.Enabled = true;
                this.dateTimePickerEndTime.Enabled = true;
                this.labelRefreshGraph.Visible = false;
                OnGraphUpdateRequested(e);
            }
        }

        private void radioButtonSelectDate_CheckedChanged(object sender, EventArgs e)
        {
            if (this.radioButtonSelectDate.Checked == true)
            {
                this.dateTimePickerStartTime.Enabled = false;
                this.dateTimePickerEndTime.Enabled = false;
                this.labelRefreshGraph.Visible = false;
                OnGraphUpdateRequested(e);
            }
            this.monthCalendarGraphDate.Enabled = this.radioButtonSelectDate.Checked;
        }

        DateTime OldMonthCalendarSelectionStart = DateTime.MinValue; // Stores previous start date selected by the month calendar
        DateTime OldMonthCalendarSelectionEnd = DateTime.MinValue;   // Stores previous end   date selected by the month calendar
        private void monthCalendarGraphDate_DateChanged(object sender, DateRangeEventArgs e)
        {
            // The DateChanged event fires every 2 minutes...
            // ...and several times when month next/previous pressed!
            // Mask this if control is not focused and if selected date actually changed.

            if ((this.monthCalendarGraphDate.Focused) && ((e.Start.CompareTo(OldMonthCalendarSelectionStart) != 0) || (e.End.CompareTo(OldMonthCalendarSelectionEnd) != 0)))
            {
                OnGraphUpdateRequested(e);
                OldMonthCalendarSelectionStart = e.Start;
                OldMonthCalendarSelectionEnd = e.End;
            }
        }

        private void monthCalendarGraphDate_EnabledChanged(object sender, EventArgs e)
        {
            if (this.monthCalendarGraphDate.Enabled == false)
            {
                this.monthCalendarGraphDate.TitleBackColor = SystemColors.InactiveCaption;
                this.monthCalendarGraphDate.TitleForeColor = SystemColors.InactiveCaptionText;
                this.monthCalendarGraphDate.ForeColor = SystemColors.GrayText;
                this.monthCalendarGraphDate.BackColor = SystemColors.Control;
            }
            else
            {
                this.monthCalendarGraphDate.TitleBackColor = SystemColors.ActiveCaption;
                this.monthCalendarGraphDate.TitleForeColor = SystemColors.ActiveCaptionText;
                this.monthCalendarGraphDate.ForeColor = SystemColors.WindowText;
                this.monthCalendarGraphDate.BackColor = SystemColors.Window;
            }
        }

        private void dateTimePickerEndTime_ValueChanged(object sender, EventArgs e)
        {
            dateTimePickerStartTime.MaxDate = dateTimePickerEndTime.Value;
            labelRefreshGraph.Visible = true;
        }

        private void dateTimePickerStartTime_ValueChanged(object sender, EventArgs e)
        {
            dateTimePickerEndTime.MinDate = dateTimePickerStartTime.Value;
            labelRefreshGraph.Visible = true;
        }

        #endregion

        #region Events
        /// <summary>
        /// GraphUpdateRequested Event
        /// </summary>

        public event EventHandler GraphUpdateRequested;  // Graph Update is requested

        protected virtual void OnGraphUpdateRequested(EventArgs e)
        {
            EventHandler handler = GraphUpdateRequested;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }
        #endregion



    }
}
